<?php
/**
 * WordPress Rewrite API
 *
 * @package WordPress
 * @subpackage Rewrite
 */

/**
 * Add a straight rewrite rule.
 *
 * @see WP_Rewrite::add_rule() for long description.
 * @since 2.1.0
 *
 * @param string $regex Regular Expression to match request against.
 * @param string $redirect Page to redirect to.
 * @param string $after Optional, default is 'bottom'. Where to add rule, can also be 'top'.
 */
function add_rewrite_rule($regex, $redirect, $after = 'bottom') {
	global $wp_rewrite;
	$wp_rewrite->add_rule($regex, $redirect, $after);
}

/**
 * Add a new tag (like %postname%).
 *
 * Warning: you must call this on init or earlier, otherwise the query var
 * addition stuff won't work.
 *
 * @since 2.1.0
 *
 * @param string $tagname
 * @param string $regex
 */
function add_rewrite_tag($tagname, $regex) {
	//validation
	if (strlen($tagname) < 3 || $tagname{0} != '%' || $tagname{strlen($tagname)-1} != '%') {
		return;
	}

	$qv = trim($tagname, '%');

	global $wp_rewrite, $wp;
	$wp->add_query_var($qv);
	$wp_rewrite->add_rewrite_tag($tagname, $regex, $qv . '=');
}

/**
 * Add a new feed type like /atom1/.
 *
 * @since 2.1.0
 *
 * @param string $feedname
 * @param callback $function Callback to run on feed display.
 * @return string Feed action name.
 */
function add_feed($feedname, $function) {
	global $wp_rewrite;
	if (!in_array($feedname, $wp_rewrite->feeds)) { //override the file if it is
		$wp_rewrite->feeds[] = $feedname;
	}
	$hook = 'do_feed_' . $feedname;
	// Remove default function hook
	remove_action($hook, $hook, 10, 1);
	add_action($hook, $function, 10, 1);
	return $hook;
}

/**
 * Endpoint Mask for Permalink.
 *
 * @since 2.1.0
 */
define('EP_PERMALINK', 1);

/**
 * Endpoint Mask for Attachment.
 *
 * @since 2.1.0
 */
define('EP_ATTACHMENT', 2);

/**
 * Endpoint Mask for date.
 *
 * @since 2.1.0
 */
define('EP_DATE', 4);

/**
 * Endpoint Mask for year
 *
 * @since 2.1.0
 */
define('EP_YEAR', 8);

/**
 * Endpoint Mask for month.
 *
 * @since 2.1.0
 */
define('EP_MONTH', 16);

/**
 * Endpoint Mask for day.
 *
 * @since 2.1.0
 */
define('EP_DAY', 32);

/**
 * Endpoint Mask for root.
 *
 * @since 2.1.0
 */
define('EP_ROOT', 64);

/**
 * Endpoint Mask for comments.
 *
 * @since 2.1.0
 */
define('EP_COMMENTS', 128);

/**
 * Endpoint Mask for searches.
 *
 * @since 2.1.0
 */
define('EP_SEARCH', 256);

/**
 * Endpoint Mask for categories.
 *
 * @since 2.1.0
 */
define('EP_CATEGORIES', 512);

/**
 * Endpoint Mask for tags.
 *
 * @since 2.3.0
 */
define('EP_TAGS', 1024);

/**
 * Endpoint Mask for authors.
 *
 * @since 2.1.0
 */
define('EP_AUTHORS', 2048);

/**
 * Endpoint Mask for pages.
 *
 * @since 2.1.0
 */
define('EP_PAGES', 4096);

//pseudo-places
/**
 * Endpoint Mask for default, which is nothing.
 *
 * @since 2.1.0
 */
define('EP_NONE', 0);

/**
 * Endpoint Mask for everything.
 *
 * @since 2.1.0
 */
define('EP_ALL', 8191);

/**
 * Add an endpoint, like /trackback/.
 *
 * The endpoints are added to the end of the request. So a request matching
 * "/2008/10/14/my_post/myep/", the endpoint will be "/myep/".
 *
 * Be sure to flush the rewrite rules (wp_rewrite->flush()) when your plugin gets
 * activated (register_activation_hook()) and deactivated (register_deactivation_hook())
 *
 * @since 2.1.0
 * @see WP_Rewrite::add_endpoint() Parameters and more description.
 * @uses $wp_rewrite
 *
 * @param unknown_type $name
 * @param unknown_type $places
 */
function add_rewrite_endpoint($name, $places) {
	global $wp_rewrite;
	$wp_rewrite->add_endpoint($name, $places);
}

/**
 * Filter the URL base for taxonomies.
 *
 * To remove any manually prepended /index.php/.
 *
 * @access private
 * @since 2.6.0
 * @author Mark Jaquith
 *
 * @param string $base The taxonomy base that we're going to filter
 * @return string
 */
function _wp_filter_taxonomy_base( $base ) {
	if ( !empty( $base ) ) {
		$base = preg_replace( '|^/index\.php/|', '', $base );
		$base = trim( $base, '/' );
	}
	return $base;
}

/**
 * Examine a url and try to determine the post ID it represents.
 *
 * Checks are supposedly from the hosted site blog.
 *
 * @since 1.0.0
 *
 * @param string $url Permalink to check.
 * @return int Post ID, or 0 on failure.
 */
function url_to_postid($url) {
	global $wp_rewrite;

	$url = apply_filters('url_to_postid', $url);

	// First, check to see if there is a 'p=N' or 'page_id=N' to match against
	if ( preg_match('#[?&](p|page_id|attachment_id)=(\d+)#', $url, $values) )	{
		$id = absint($values[2]);
		if ($id)
			return $id;
	}

	// Check to see if we are using rewrite rules
	$rewrite = $wp_rewrite->wp_rewrite_rules();

	// Not using rewrite rules, and 'p=N' and 'page_id=N' methods failed, so we're out of options
	if ( empty($rewrite) )
		return 0;

	// $url cleanup by Mark Jaquith
	// This fixes things like #anchors, ?query=strings, missing 'www.',
	// added 'www.', or added 'index.php/' that will mess up our WP_Query
	// and return a false negative

	// Get rid of the #anchor
	$url_split = explode('#', $url);
	$url = $url_split[0];

	// Get rid of URL ?query=string
	$url_split = explode('?', $url);
	$url = $url_split[0];

	// Add 'www.' if it is absent and should be there
	if ( false !== strpos(get_option('home'), '://www.') && false === strpos($url, '://www.') )
		$url = str_replace('://', '://www.', $url);

	// Strip 'www.' if it is present and shouldn't be
	if ( false === strpos(get_option('home'), '://www.') )
		$url = str_replace('://www.', '://', $url);

	// Strip 'index.php/' if we're not using path info permalinks
	if ( !$wp_rewrite->using_index_permalinks() )
		$url = str_replace('index.php/', '', $url);

	if ( false !== strpos($url, get_option('home')) ) {
		// Chop off http://domain.com
		$url = str_replace(get_option('home'), '', $url);
	} else {
		// Chop off /path/to/blog
		$home_path = parse_url(get_option('home'));
		$home_path = $home_path['path'];
		$url = str_replace($home_path, '', $url);
	}

	// Trim leading and lagging slashes
	$url = trim($url, '/');

	$request = $url;

	// Done with cleanup

	// Look for matches.
	$request_match = $request;
	foreach ($rewrite as $match => $query) {
		// If the requesting file is the anchor of the match, prepend it
		// to the path info.
		if ( (! empty($url)) && (strpos($match, $url) === 0) && ($url != $request)) {
			$request_match = $url . '/' . $request;
		}

		if ( preg_match("!^$match!", $request_match, $matches) ) {
			// Got a match.
			// Trim the query of everything up to the '?'.
			$query = preg_replace("!^.+\?!", '', $query);

			// Substitute the substring matches into the query.
			$query = addslashes(WP_MatchesMapRegex::apply($query, $matches));
			// Filter out non-public query vars
			global $wp;
			parse_str($query, $query_vars);
			$query = array();
			foreach ( (array) $query_vars as $key => $value ) {
				if ( in_array($key, $wp->public_query_vars) )
					$query[$key] = $value;
			}
			// Do the query
			$query = new WP_Query($query);
			if ( $query->is_single || $query->is_page )
				return $query->post->ID;
			else
				return 0;
		}
	}
	return 0;
}

/**
 * WordPress Rewrite Component.
 *
 * The WordPress Rewrite class writes the rewrite module rules to the .htaccess
 * file. It also handles parsing the request to get the correct setup for the
 * WordPress Query class.
 *
 * The Rewrite along with WP class function as a front controller for WordPress.
 * You can add rules to trigger your page view and processing using this
 * component. The full functionality of a front controller does not exist,
 * meaning you can't define how the template files load based on the rewrite
 * rules.
 *
 * @since 1.5.0
 */
class WP_Rewrite {
	/**
	 * Default permalink structure for WordPress.
	 *
	 * @since 1.5.0
	 * @access private
	 * @var string
	 */
	var $permalink_structure;

	/**
	 * Whether to add trailing slashes.
	 *
	 * @since 2.2.0
	 * @access private
	 * @var bool
	 */
	var $use_trailing_slashes;

	/**
	 * Customized or default category permalink base ( example.com/xx/tagname ).
	 *
	 * @since 1.5.0
	 * @access private
	 * @var string
	 */
	var $category_base;

	/**
	 * Customized or default tag permalink base ( example.com/xx/tagname ).
	 *
	 * @since 2.3.0
	 * @access private
	 * @var string
	 */
	var $tag_base;

	/**
	 * Permalink request structure for categories.
	 *
	 * @since 1.5.0
	 * @access private
	 * @var string
	 */
	var $category_structure;

	/**
	 * Permalink request structure for tags.
	 *
	 * @since 2.3.0
	 * @access private
	 * @var string
	 */
	var $tag_structure;

	/**
	 * Permalink author request base ( example.com/author/authorname ).
	 *
	 * @since 1.5.0
	 * @access private
	 * @var string
	 */
	var $author_base = 'author';

	/**
	 * Permalink request structure for author pages.
	 *
	 * @since 1.5.0
	 * @access private
	 * @var string
	 */
	var $author_structure;

	/**
	 * Permalink request structure for dates.
	 *
	 * @since 1.5.0
	 * @access private
	 * @var string
	 */
	var $date_structure;

	/**
	 * Permalink request structure for pages.
	 *
	 * @since 1.5.0
	 * @access private
	 * @var string
	 */
	var $page_structure;

	/**
	 * Search permalink base ( example.com/search/query ).
	 *
	 * @since 1.5.0
	 * @access private
	 * @var string
	 */
	var $search_base = 'search';

	/**
	 * Permalink request structure for searches.
	 *
	 * @since 1.5.0
	 * @access private
	 * @var string
	 */
	var $search_structure;

	/**
	 * Comments permalink base.
	 *
	 * @since 1.5.0
	 * @access private
	 * @var string
	 */
	var $comments_base = 'comments';

	/**
	 * Feed permalink base.
	 *
	 * @since 1.5.0
	 * @access private
	 * @var string
	 */
	var $feed_base = 'feed';

	/**
	 * Comments feed request structure permalink.
	 *
	 * @since 1.5.0
	 * @access private
	 * @var string
	 */
	var $comments_feed_structure;

	/**
	 * Feed request structure permalink.
	 *
	 * @since 1.5.0
	 * @access private
	 * @var string
	 */
	var $feed_structure;

	/**
	 * Front URL path.
	 *
	 * The difference between the root property is that WordPress might be
	 * located at example/WordPress/index.php, if permalinks are turned off. The
	 * WordPress/index.php will be the front portion. If permalinks are turned
	 * on, this will most likely be empty or not set.
	 *
	 * @since 1.5.0
	 * @access private
	 * @var string
	 */
	var $front;

	/**
	 * Root URL path to WordPress (without domain).
	 *
	 * The difference between front property is that WordPress might be located
	 * at example.com/WordPress/. The root is the 'WordPress/' portion.
	 *
	 * @since 1.5.0
	 * @access private
	 * @var string
	 */
	var $root = '';

	/**
	 * Permalink to the home page.
	 *
	 * @since 1.5.0
	 * @access public
	 * @var string
	 */
	var $index = 'index.php';

	/**
	 * Request match string.
	 *
	 * @since 1.5.0
	 * @access private
	 * @var string
	 */
	var $matches = '';

	/**
	 * Rewrite rules to match against the request to find the redirect or query.
	 *
	 * @since 1.5.0
	 * @access private
	 * @var array
	 */
	var $rules;

	/**
	 * Additional rules added external to the rewrite class.
	 *
	 * Those not generated by the class, see add_rewrite_rule().
	 *
	 * @since 2.1.0
	 * @access private
	 * @var array
	 */
	var $extra_rules = array(); //

	/**
	 * Additional rules that belong at the beginning to match first.
	 *
	 * Those not generated by the class, see add_rewrite_rule().
	 *
	 * @since 2.3.0
	 * @access private
	 * @var array
	 */
	var $extra_rules_top = array(); //

	/**
	 * Rules that don't redirect to WP's index.php.
	 *
	 * These rules are written to the mod_rewrite portion of the .htaccess.
	 *
	 * @since 2.1.0
	 * @access private
	 * @var array
	 */
	var $non_wp_rules = array(); //

	/**
	 * Extra permalink structures.
	 *
	 * @since 2.1.0
	 * @access private
	 * @var array
	 */
	var $extra_permastructs = array();

	/**
	 * Endpoints permalinks
	 *
	 * @since unknown
	 * @access private
	 * @var array
	 */
	var $endpoints;

	/**
	 * Whether to write every mod_rewrite rule for WordPress.
	 *
	 * This is off by default, turning it on might print a lot of rewrite rules
	 * to the .htaccess file.
	 *
	 * @since 2.0.0
	 * @access public
	 * @var bool
	 */
	var $use_verbose_rules = false;

	/**
	 * Whether to write every mod_rewrite rule for WordPress pages.
	 *
	 * @since 2.5.0
	 * @access public
	 * @var bool
	 */
	var $use_verbose_page_rules = true;

	/**
	 * Permalink structure search for preg_replace.
	 *
	 * @since 1.5.0
	 * @access private
	 * @var array
	 */
	var $rewritecode =
		array(
					'%year%',
					'%monthnum%',
					'%day%',
					'%hour%',
					'%minute%',
					'%second%',
					'%postname%',
					'%post_id%',
					'%category%',
					'%tag%',
					'%author%',
					'%pagename%',
					'%search%'
					);

	/**
	 * Preg_replace values for the search, see {@link WP_Rewrite::$rewritecode}.
	 *
	 * @since 1.5.0
	 * @access private
	 * @var array
	 */
	var $rewritereplace =
		array(
					'([0-9]{4})',
					'([0-9]{1,2})',
					'([0-9]{1,2})',
					'([0-9]{1,2})',
					'([0-9]{1,2})',
					'([0-9]{1,2})',
					'([^/]+)',
					'([0-9]+)',
					'(.+?)',
					'(.+?)',
					'([^/]+)',
					'([^/]+?)',
					'(.+)'
					);

	/**
	 * Search for the query to look for replacing.
	 *
	 * @since 1.5.0
	 * @access private
	 * @var array
	 */
	var $queryreplace =
		array (
					'year=',
					'monthnum=',
					'day=',
					'hour=',
					'minute=',
					'second=',
					'name=',
					'p=',
					'category_name=',
					'tag=',
					'author_name=',
					'pagename=',
					's='
					);

	/**
	 * Supported default feeds.
	 *
	 * @since 1.5.0
	 * @access private
	 * @var array
	 */
	var $feeds = array ( 'feed', 'rdf', 'rss', 'rss2', 'atom' );

	/**
	 * Whether permalinks are being used.
	 *
	 * This can be either rewrite module or permalink in the HTTP query string.
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @return bool True, if permalinks are enabled.
	 */
	function using_permalinks() {
		if (empty($this->permalink_structure))
			return false;
		else
			return true;
	}

	/**
	 * Whether permalinks are being used and rewrite module is not enabled.
	 *
	 * Means that permalink links are enabled and index.php is in the URL.
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @return bool
	 */
	function using_index_permalinks() {
		if (empty($this->permalink_structure)) {
			return false;
		}

		// If the index is not in the permalink, we're using mod_rewrite.
		if (preg_match('#^/*' . $this->index . '#', $this->permalink_structure)) {
			return true;
		}

		return false;
	}

	/**
	 * Whether permalinks are being used and rewrite module is enabled.
	 *
	 * Using permalinks and index.php is not in the URL.
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @return bool
	 */
	function using_mod_rewrite_permalinks() {
		if ( $this->using_permalinks() && ! $this->using_index_permalinks())
			return true;
		else
			return false;
	}

	/**
	 * Index for matches for usage in preg_*() functions.
	 *
	 * The format of the string is, with empty matches property value, '$NUM'.
	 * The 'NUM' will be replaced with the value in the $number parameter. With
	 * the matches property not empty, the value of the returned string will
	 * contain that value of the matches property. The format then will be
	 * '$MATCHES[NUM]', with MATCHES as the value in the property and NUM the
	 * value of the $number parameter.
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @param int $number Index number.
	 * @return string
	 */
	function preg_index($number) {
		$match_prefix = '$';
		$match_suffix = '';

		if ( ! empty($this->matches) ) {
			$match_prefix = '$' . $this->matches . '[';
			$match_suffix = ']';
		}

		return "$match_prefix$number$match_suffix";
	}

	/**
	 * Retrieve all page and attachments for pages URIs.
	 *
	 * The attachments are for those that have pages as parents and will be
	 * retrieved.
	 *
	 * @since 2.5.0
	 * @access public
	 *
	 * @return array Array of page URIs as first element and attachment URIs as second element.
	 */
	function page_uri_index() {
		global $wpdb;

		//get pages in order of hierarchy, i.e. children after parents
		$posts = get_page_hierarchy($wpdb->get_results("SELECT ID, post_name, post_parent FROM $wpdb->posts WHERE post_type = 'page'"));
		//now reverse it, because we need parents after children for rewrite rules to work properly
		$posts = array_reverse($posts, true);

		$page_uris = array();
		$page_attachment_uris = array();

		if ( !$posts )
			return array( array(), array() );

		foreach ($posts as $id => $post) {
			// URL => page name
			$uri = get_page_uri($id);
			$attachments = $wpdb->get_results( $wpdb->prepare( "SELECT ID, post_name, post_parent FROM $wpdb->posts WHERE post_type = 'attachment' AND post_parent = %d", $id ));
			if ( $attachments ) {
				foreach ( $attachments as $attachment ) {
					$attach_uri = get_page_uri($attachment->ID);
					$page_attachment_uris[$attach_uri] = $attachment->ID;
				}
			}

			$page_uris[$uri] = $id;
		}

		return array( $page_uris, $page_attachment_uris );
	}

	/**
	 * Retrieve all of the rewrite rules for pages.
	 *
	 * If the 'use_verbose_page_rules' property is false, then there will only
	 * be a single rewrite rule for pages for those matching '%pagename%'. With
	 * the property set to true, the attachments and the pages will be added for
	 * each individual attachment URI and page URI, respectively.
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @return array
	 */
	function page_rewrite_rules() {
		$rewrite_rules = array();
		$page_structure = $this->get_page_permastruct();

		if ( ! $this->use_verbose_page_rules ) {
			$this->add_rewrite_tag('%pagename%', "(.+?)", 'pagename=');
			$rewrite_rules = array_merge($rewrite_rules, $this->generate_rewrite_rules($page_structure, EP_PAGES));
			return $rewrite_rules;
		}

		$page_uris = $this->page_uri_index();
		$uris = $page_uris[0];
		$attachment_uris = $page_uris[1];

		if( is_array( $attachment_uris ) ) {
			foreach ($attachment_uris as $uri => $pagename) {
				$this->add_rewrite_tag('%pagename%', "($uri)", 'attachment=');
				$rewrite_rules = array_merge($rewrite_rules, $this->generate_rewrite_rules($page_structure, EP_PAGES));
			}
		}
		if( is_array( $uris ) ) {
			foreach ($uris as $uri => $pagename) {
				$this->add_rewrite_tag('%pagename%', "($uri)", 'pagename=');
				$rewrite_rules = array_merge($rewrite_rules, $this->generate_rewrite_rules($page_structure, EP_PAGES));
			}
		}

		return $rewrite_rules;
	}

	/**
	 * Retrieve date permalink structure, with year, month, and day.
	 *
	 * The permalink structure for the date, if not set already depends on the
	 * permalink structure. It can be one of three formats. The first is year,
	 * month, day; the second is day, month, year; and the last format is month,
	 * day, year. These are matched against the permalink structure for which
	 * one is used. If none matches, then the default will be used, which is
	 * year, month, day.
	 *
	 * Prevents post ID and date permalinks from overlapping. In the case of
	 * post_id, the date permalink will be prepended with front permalink with
	 * 'date/' before the actual permalink to form the complete date permalink
	 * structure.
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @return bool|string False on no permalink structure. Date permalink structure.
	 */
	function get_date_permastruct() {
		if (isset($this->date_structure)) {
			return $this->date_structure;
		}

		if (empty($this->permalink_structure)) {
			$this->date_structure = '';
			return false;
		}

		// The date permalink must have year, month, and day separated by slashes.
		$endians = array('%year%/%monthnum%/%day%', '%day%/%monthnum%/%year%', '%monthnum%/%day%/%year%');

		$this->date_structure = '';
		$date_endian = '';

		foreach ($endians as $endian) {
			if (false !== strpos($this->permalink_structure, $endian)) {
				$date_endian= $endian;
				break;
			}
		}

		if ( empty($date_endian) )
			$date_endian = '%year%/%monthnum%/%day%';

		// Do not allow the date tags and %post_id% to overlap in the permalink
		// structure. If they do, move the date tags to $front/date/.
		$front = $this->front;
		preg_match_all('/%.+?%/', $this->permalink_structure, $tokens);
		$tok_index = 1;
		foreach ( (array) $tokens[0] as $token) {
			if ( ($token == '%post_id%') && ($tok_index <= 3) ) {
				$front = $front . 'date/';
				break;
			}
			$tok_index++;
		}

		$this->date_structure = $front . $date_endian;

		return $this->date_structure;
	}

	/**
	 * Retrieve the year permalink structure without month and day.
	 *
	 * Gets the date permalink structure and strips out the month and day
	 * permalink structures.
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @return bool|string False on failure. Year structure on success.
	 */
	function get_year_permastruct() {
		$structure = $this->get_date_permastruct($this->permalink_structure);

		if (empty($structure)) {
			return false;
		}

		$structure = str_replace('%monthnum%', '', $structure);
		$structure = str_replace('%day%', '', $structure);

		$structure = preg_replace('#/+#', '/', $structure);

		return $structure;
	}

	/**
	 * Retrieve the month permalink structure without day and with year.
	 *
	 * Gets the date permalink structure and strips out the day permalink
	 * structures. Keeps the year permalink structure.
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @return bool|string False on failure. Year/Month structure on success.
	 */
	function get_month_permastruct() {
		$structure = $this->get_date_permastruct($this->permalink_structure);

		if (empty($structure)) {
			return false;
		}

		$structure = str_replace('%day%', '', $structure);

		$structure = preg_replace('#/+#', '/', $structure);

		return $structure;
	}

	/**
	 * Retrieve the day permalink structure with month and year.
	 *
	 * Keeps date permalink structure with all year, month, and day.
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @return bool|string False on failure. Year/Month/Day structure on success.
	 */
	function get_day_permastruct() {
		return $this->get_date_permastruct($this->permalink_structure);
	}

	/**
	 * Retrieve the permalink structure for categories.
	 *
	 * If the category_base property has no value, then the category structure
	 * will have the front property value, followed by 'category', and finally
	 * '%category%'. If it does, then the root property will be used, along with
	 * the category_base property value.
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @return bool|string False on failure. Category permalink structure.
	 */
	function get_category_permastruct() {
		if (isset($this->category_structure)) {
			return $this->category_structure;
		}

		if (empty($this->permalink_structure)) {
			$this->category_structure = '';
			return false;
		}

		if (empty($this->category_base))
			$this->category_structure = trailingslashit( $this->front . 'category' );
		else
			$this->category_structure = trailingslashit( '/' . $this->root . $this->category_base );

		$this->category_structure .= '%category%';

		return $this->category_structure;
	}

	/**
	 * Retrieve the permalink structure for tags.
	 *
	 * If the tag_base property has no value, then the tag structure will have
	 * the front property value, followed by 'tag', and finally '%tag%'. If it
	 * does, then the root property will be used, along with the tag_base
	 * property value.
	 *
	 * @since 2.3.0
	 * @access public
	 *
	 * @return bool|string False on failure. Tag permalink structure.
	 */
	function get_tag_permastruct() {
		if (isset($this->tag_structure)) {
			return $this->tag_structure;
		}

		if (empty($this->permalink_structure)) {
			$this->tag_structure = '';
			return false;
		}

		if (empty($this->tag_base))
			$this->tag_structure = trailingslashit( $this->front . 'tag' );
		else
			$this->tag_structure = trailingslashit( '/' . $this->root . $this->tag_base );

		$this->tag_structure .= '%tag%';

		return $this->tag_structure;
	}

	/**
	 * Retrieve extra permalink structure by name.
	 *
	 * @since unknown
	 * @access public
	 *
	 * @param string $name Permalink structure name.
	 * @return string|bool False if not found. Permalink structure string.
	 */
	function get_extra_permastruct($name) {
		if ( empty($this->permalink_structure) )
			return false;
		if ( isset($this->extra_permastructs[$name]) )
			return $this->extra_permastructs[$name];
		return false;
	}

	/**
	 * Retrieve the author permalink structure.
	 *
	 * The permalink structure is front property, author base, and finally
	 * '/%author%'. Will set the author_structure property and then return it
	 * without attempting to set the value again.
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @return string|bool False if not found. Permalink structure string.
	 */
	function get_author_permastruct() {
		if (isset($this->author_structure)) {
			return $this->author_structure;
		}

		if (empty($this->permalink_structure)) {
			$this->author_structure = '';
			return false;
		}

		$this->author_structure = $this->front . $this->author_base . '/%author%';

		return $this->author_structure;
	}

	/**
	 * Retrieve the search permalink structure.
	 *
	 * The permalink structure is root property, search base, and finally
	 * '/%search%'. Will set the search_structure property and then return it
	 * without attempting to set the value again.
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @return string|bool False if not found. Permalink structure string.
	 */
	function get_search_permastruct() {
		if (isset($this->search_structure)) {
			return $this->search_structure;
		}

		if (empty($this->permalink_structure)) {
			$this->search_structure = '';
			return false;
		}

		$this->search_structure = $this->root . $this->search_base . '/%search%';

		return $this->search_structure;
	}

	/**
	 * Retrieve the page permalink structure.
	 *
	 * The permalink structure is root property, and '%pagename%'. Will set the
	 * page_structure property and then return it without attempting to set the
	 * value again.
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @return string|bool False if not found. Permalink structure string.
	 */
	function get_page_permastruct() {
		if (isset($this->page_structure)) {
			return $this->page_structure;
		}

		if (empty($this->permalink_structure)) {
			$this->page_structure = '';
			return false;
		}

		$this->page_structure = $this->root . '%pagename%';

		return $this->page_structure;
	}

	/**
	 * Retrieve the feed permalink structure.
	 *
	 * The permalink structure is root property, feed base, and finally
	 * '/%feed%'. Will set the feed_structure property and then return it
	 * without attempting to set the value again.
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @return string|bool False if not found. Permalink structure string.
	 */
	function get_feed_permastruct() {
		if (isset($this->feed_structure)) {
			return $this->feed_structure;
		}

		if (empty($this->permalink_structure)) {
			$this->feed_structure = '';
			return false;
		}

		$this->feed_structure = $this->root . $this->feed_base . '/%feed%';

		return $this->feed_structure;
	}

	/**
	 * Retrieve the comment feed permalink structure.
	 *
	 * The permalink structure is root property, comment base property, feed
	 * base and finally '/%feed%'. Will set the comment_feed_structure property
	 * and then return it without attempting to set the value again.
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @return string|bool False if not found. Permalink structure string.
	 */
	function get_comment_feed_permastruct() {
		if (isset($this->comment_feed_structure)) {
			return $this->comment_feed_structure;
		}

		if (empty($this->permalink_structure)) {
			$this->comment_feed_structure = '';
			return false;
		}

		$this->comment_feed_structure = $this->root . $this->comments_base . '/' . $this->feed_base . '/%feed%';

		return $this->comment_feed_structure;
	}

	/**
	 * Append or update tag, pattern, and query for replacement.
	 *
	 * If the tag already exists, replace the existing pattern and query for
	 * that tag, otherwise add the new tag, pattern, and query to the end of the
	 * arrays.
	 *
	 * @internal What is the purpose of this function again? Need to finish long
	 *           description.
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @param string $tag Append tag to rewritecode property array.
	 * @param string $pattern Append pattern to rewritereplace property array.
	 * @param string $query Append query to queryreplace property array.
	 */
	function add_rewrite_tag($tag, $pattern, $query) {
		$position = array_search($tag, $this->rewritecode);
		if ( false !== $position && null !== $position ) {
			$this->rewritereplace[$position] = $pattern;
			$this->queryreplace[$position] = $query;
		} else {
			$this->rewritecode[] = $tag;
			$this->rewritereplace[] = $pattern;
			$this->queryreplace[] = $query;
		}
	}

	/**
	 * Generate the rules from permalink structure.
	 *
	 * The main WP_Rewrite function for building the rewrite rule list. The
	 * contents of the function is a mix of black magic and regular expressions,
	 * so best just ignore the contents and move to the parameters.
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @param string $permalink_structure The permalink structure.
	 * @param int $ep_mask Optional, default is EP_NONE. Endpoint constant, see EP_* constants.
	 * @param bool $paged Optional, default is true. Whether permalink request is paged.
	 * @param bool $feed Optional, default is true. Whether for feed.
	 * @param bool $forcomments Optional, default is false. Whether for comments.
	 * @param bool $walk_dirs Optional, default is true. Whether to create list of directories to walk over.
	 * @param bool $endpoints Optional, default is true. Whether endpoints are enabled.
	 * @return array Rewrite rule list.
	 */
	function generate_rewrite_rules($permalink_structure, $ep_mask = EP_NONE, $paged = true, $feed = true, $forcomments = false, $walk_dirs = true, $endpoints = true) {
		//build a regex to match the feed section of URLs, something like (feed|atom|rss|rss2)/?
		$feedregex2 = '';
		foreach ( (array) $this->feeds as $feed_name) {
			$feedregex2 .= $feed_name . '|';
		}
		$feedregex2 = '(' . trim($feedregex2, '|') .  ')/?$';
		//$feedregex is identical but with /feed/ added on as well, so URLs like <permalink>/feed/atom
		//and <permalink>/atom are both possible
		$feedregex = $this->feed_base  . '/' . $feedregex2;

		//build a regex to match the trackback and page/xx parts of URLs
		$trackbackregex = 'trackback/?$';
		$pageregex = 'page/?([0-9]{1,})/?$';
		$commentregex = 'comment-page-([0-9]{1,})/?$';

		//build up an array of endpoint regexes to append => queries to append
		if ($endpoints) {
			$ep_query_append = array ();
			foreach ( (array) $this->endpoints as $endpoint) {
				//match everything after the endpoint name, but allow for nothing to appear there
				$epmatch = $endpoint[1] . '(/(.*))?/?$';
				//this will be appended on to the rest of the query for each dir
				$epquery = '&' . $endpoint[1] . '=';
				$ep_query_append[$epmatch] = array ( $endpoint[0], $epquery );
			}
		}

		//get everything up to the first rewrite tag
		$front = substr($permalink_structure, 0, strpos($permalink_structure, '%'));
		//build an array of the tags (note that said array ends up being in $tokens[0])
		preg_match_all('/%.+?%/', $permalink_structure, $tokens);

		$num_tokens = count($tokens[0]);

		$index = $this->index; //probably 'index.php'
		$feedindex = $index;
		$trackbackindex = $index;
		//build a list from the rewritecode and queryreplace arrays, that will look something like
		//tagname=$matches[i] where i is the current $i
		for ($i = 0; $i < $num_tokens; ++$i) {
			if (0 < $i) {
				$queries[$i] = $queries[$i - 1] . '&';
			} else {
				$queries[$i] = '';
			}

			$query_token = str_replace($this->rewritecode, $this->queryreplace, $tokens[0][$i]) . $this->preg_index($i+1);
			$queries[$i] .= $query_token;
		}

		//get the structure, minus any cruft (stuff that isn't tags) at the front
		$structure = $permalink_structure;
		if ($front != '/') {
			$structure = str_replace($front, '', $structure);
		}
		//create a list of dirs to walk over, making rewrite rules for each level
		//so for example, a $structure of /%year%/%month%/%postname% would create
		//rewrite rules for /%year%/, /%year%/%month%/ and /%year%/%month%/%postname%
		$structure = trim($structure, '/');
		if ($walk_dirs) {
			$dirs = explode('/', $structure);
		} else {
			$dirs[] = $structure;
		}
		$num_dirs = count($dirs);

		//strip slashes from the front of $front
		$front = preg_replace('|^/+|', '', $front);

		//the main workhorse loop
		$post_rewrite = array();
		$struct = $front;
		for ($j = 0; $j < $num_dirs; ++$j) {
			//get the struct for this dir, and trim slashes off the fr